/**
 * External dependencies
 */
import classnames from 'classnames';

/**
 * WordPress dependencies
 */
import { __, sprintf } from '@wordpress/i18n';
import { dateI18n } from '@wordpress/date';
import { Button, Modal } from '@wordpress/components';
import { compose, withState, withInstanceId } from '@wordpress/compose';

/**
 * Internal dependencies
 */
import { getConfigValue } from '../../utils';
import ScanOnly from './scan-only';
import Header, { Title, Date } from '../../components/card/header';
import { FooterSchemaActions } from '../../components/card/footer';
import { MalwareScanResults, PrintR } from 'packages/components/src';
import './style.scss';

function getScanStatus( entry ) {
	switch ( entry.code ) {
		case 'scan-failure-server-error':
		case 'scan-failure-client-error':
		case 'sucuri-system-error':
			return [ 'warn', __( 'Scan Error', 'ithemes-security-pro' ) ];
		case 'found-malware-and-on-blacklist':
			return [ 'error', __( 'Found Malware, On Blacklist', 'ithemes-security-pro' ) ];
		case 'found-malware':
			return [ 'error', __( 'Found Malware', 'ithemes-security-pro' ) ];
		case 'on-blacklist':
			return [ 'error', __( 'On Blacklist', 'ithemes-security-pro' ) ];
		case 'clean':
			return [ 'clean', __( 'Clean', 'ithemes-security-pro' ) ];
		default:
			return [ 'unknown', entry.code ];
	}
}

function MalwareScan( { card, config, viewEntry, scanResults, showRawDetails, setState, instanceId } ) {
	return (
		<div className="itsec-card--type-malware-scan">
			<Header>
				<Title card={ card } config={ config } />
				<Date card={ card } config={ config } />
			</Header>
			<section className="itsec-card-malware-scan__scans-section">
				<table className="itsec-card-malware-scan__scans">
					<thead>
						<tr>
							<th>{ __( 'Time', 'ithemes-security-pro' ) }</th>
							<th>{ __( 'Status', 'ithemes-security-pro' ) }</th>
							<th><span className="screen-reader-text">{ __( 'Actions', 'ithemes-security-pro' ) }</span></th>
						</tr>
					</thead>
					<tbody>
						{ card.data.map( ( entry ) => {
							const [ status, label ] = getScanStatus( entry );

							return (
								<tr key={ entry.id }>
									<th scope="row">{ dateI18n( 'M d, Y g:s A', entry.time ) }</th>
									<td>
										<span className={ classnames( 'itsec-card-malware-scan__scan-status', `itsec-card-malware-scan__scan-status--${ status }` ) }>
											{ label }
										</span>
									</td>
									<td>
										<Button isLink aria-pressed={ viewEntry === entry.id } onClick={ () => setState( { viewEntry: entry.id } ) }>
											{ __( 'View', 'ithemes-security-pro' ) }
										</Button>
										{ viewEntry === entry.id && (
											<Modal
												title={ sprintf( __( 'View Scan Details for %s', 'ithemes-security-pro' ), dateI18n( 'M d, Y g:s A', entry.time ) ) }
												onRequestClose={ () => setState( { viewEntry: 0, showRawDetails: false } ) }
											>
												<MalwareScanResults results={ entry.results } />
												<Button className="itsec-card-malware-scan__raw-details-toggle" isLink onClick={ () => setState( { showRawDetails: ! showRawDetails } ) }
													aria-expanded={ showRawDetails } aria-controls={ `itsec-card-malware-scan__raw-details--${ instanceId }` }>
													{ showRawDetails ? __( 'Hide Raw Details', 'ithemes-security-pro' ) : __( 'Show Raw Details', 'ithemes-security-pro' ) }
												</Button>
												<div id={ `itsec-card-malware-scan__raw-details--${ instanceId }` } style={ { visibility: showRawDetails ? 'visible' : 'hidden' } }>
													{ showRawDetails && <PrintR json={ entry.results } /> }
												</div>
											</Modal>
										) }
									</td>
								</tr>
							);
						} ) }
					</tbody>
				</table>
			</section>
			<FooterSchemaActions card={ card } onComplete={ ( href, response ) => href.endsWith( '/scan' ) && setState( { scanResults: response.results } ) } />
			{ scanResults && (
				<Modal
					title={ __( 'Scan Results', 'ithemes-security-pro' ) }
					onRequestClose={ () => setState( { scanResults: undefined } ) }
				>
					<MalwareScanResults results={ scanResults } />
				</Modal>
			) }
		</div>
	);
}

export const slug = 'malware-scan';
export const settings = {
	render: getConfigValue( 'db_logs' ) ? compose( [
		withState( { viewEntry: 0, scanResults: undefined, showRawDetails: false } ),
		withInstanceId,
	] )( MalwareScan ) : ScanOnly,
};
